package com.vmware.samples.chassisui.views {

import com.vmware.samples.chassisui.Util;
import com.vmware.samples.chassisui.WizardCompletionEvent;
import com.vmware.samples.chassisui.model.ChassisInfo;
import com.vmware.samples.chassisui.views.wizardPages.DimensionsPage;
import com.vmware.samples.chassisui.views.wizardPages.NamePage;
import com.vmware.samples.chassisui.views.wizardPages.SummaryPage;
import com.vmware.ui.wizard.AppWizard;
import com.vmware.widgets.controls.wizardclasses.PageInfo;
import com.vmware.widgets.controls.wizardclasses.PageState;
import com.vmware.widgets.events.DialogEvent;

import mx.binding.utils.BindingUtils;
import mx.collections.ArrayCollection;
import mx.core.UIComponent;

[Event(name="{com.vmware.samples.chassisui.WizardCompletionEvent.EVENT_ID}",
        type="com.vmware.samples.chassisui.WizardCompletionEvent")]
/**
 * Wizard for creating a new Chassis object.
 *
 * This sample contains 3 pages and it demonstrates the basics of the wizard APIs:
 * - Use of a common data model
 * - Addition of pages
 * - Initialization requiring a server call
 * - Client-side and server-side validation
 * - Data persistence across sessions if the wizard is left in "Work in Progress"
 *
 * -------------------------------------------------------------------------------
 * IMPORTANT: As indicated in the SDK release notes the Wizard APIs are provided
 * as "internal APIs" that are not part of the SDK itself, in the sense that they
 * are not guaranteed to be compatible with the next major SDK release. You can
 * copy this code to use the Wizard APIs in 5.5, but you will likely have to
 * update your plugin before it can run with the next major release after 5.5.
 * -------------------------------------------------------------------------------
 */
public class ChassisWizard extends AppWizard {

   // Wizard pages
   static private const NAME_PAGE_ID:String = "chassisNamePage";
   static private const DIMENSIONS_PAGE_ID:String = "chassisDimensionPage";
   static private const SUMMARY_PAGE_ID:String = "summaryPage";
   private var _namePage:PageInfo;
   private var _dimensionsPage:PageInfo;
   private var _summaryPage:PageInfo;

   private var _pagesBeingCreated:Boolean = false;

   /**
    * ChassisWizard contructor.
    *
    * Note: passing any argument to the constructor (proxy, callback, etc.) will break
    * the wizard persistence feature because serialization requires an empty constructor.
    */
   public function ChassisWizard() {

      // You can adjust the container size if the default size is too big but you should
      // avoid having pages with scroll bars or variable sizes.
      UIComponent(this.container).minHeight = 400;
      UIComponent(this.container).minWidth = 700;

      // Handle the wizard completion (click on the Finish button)
      this.addEventListener(DialogEvent.OK, onWizardCompleted, false, 0, true);

      // Start with an empty data model, each page will do its own initialization
      this.data = new ChassisInfo();

      addPages();
   }

   /**
    * Data model shared with all pages.
    * [Inheritable] is required for that property to be passed to the page mediators.
    * [Persistable] is required for the wizard to persist across user sessions.
    */
   private var _data:ChassisInfo;

   [Inheritable]
   [Bindable]
   [Persistable]
   public function set data(value:ChassisInfo):void {
      if (_data != value) {
         _data = value;
      }
   }

   /** @private */
   public function get data():ChassisInfo {
      return _data;
   }

   /**
    * Create the initial collection of PageInfo's in the desired order.
    *
    * The first page will be displayed when the ChassisWizard constructor returns.
    * A more advanced wizard could have additional PageInfo's created dynamically
    * depending on the user's selections.
    */
   private function addPages():void {
      _pagesBeingCreated = true;
      var pageInfoCollection:ArrayCollection = new ArrayCollection();

      _namePage = createPage(new NamePage(), NAME_PAGE_ID,
         Util.getString("wizard.namePage.title"),
         Util.getString("wizard.namePage.description"));
      pageInfoCollection.addItem(_namePage);
      _namePage.state = PageState.INCOMPLETE;

      _dimensionsPage = createPage(new DimensionsPage(), DIMENSIONS_PAGE_ID,
         Util.getString("wizard.dimensionsPage.title"),
         Util.getString("wizard.dimensionsPage.description"));
      pageInfoCollection.addItem(_dimensionsPage);
      _dimensionsPage.state = PageState.INCOMPLETE;

      _summaryPage = createPage(new SummaryPage(), SUMMARY_PAGE_ID,
         Util.getString("wizard.summaryPage.title"),
         Util.getString("wizard.summaryPage.description"));
      _summaryPage.state = PageState.DISABLED;
      pageInfoCollection.addItem(_summaryPage);

      this.pagesInfo = pageInfoCollection;
      _pagesBeingCreated = false;
   }

   private function createPage(page:UIComponent, pageId:String, pageTitle:String,
                               pageDescription:String):PageInfo {
      page.id = pageId;
      var pageInfo:PageInfo = new PageInfo(page, pageTitle, pageDescription);
      BindingUtils.bindSetter(onPageStateChanged, pageInfo, "state");
      return pageInfo;
   }

   /**
    * Handles changes in page states.
    */
   private function onPageStateChanged(value:Object):void {
      if (_pagesBeingCreated) {
         // Ignore events generated during addPages()
         return;
      }
      // The last page (summary) can be reached only if namePage is complete.
      // The dimensionPage can stay incomplete because dimensions are optional.
      if (_namePage.state == PageState.COMPLETE) {
         _summaryPage.state = PageState.INCOMPLETE;
      } else {
         _summaryPage.state = PageState.DISABLED;
      }
   }

   /**
    * Callback for the Finish button.
    */
   private function onWizardCompleted(event:DialogEvent):void {
      // Send the data back to the caller (in this case ChassisActionCommand)
      //
      // Note: calling the Java service proxy here is not recommended because the
      // response handler onCreateChassisComplete would not be able to dispatch any
      // ModelChangeEvent as the ChassisWizard is already removed from the event bus.

      dispatchEvent(WizardCompletionEvent.newInstance(data, "ChassisWizard"));
   }
}
}